"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthManager = exports.AUTH_TYPE_NAMES = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _coreHttpServer = require("@kbn/core-http-server");
var _lodash = require("lodash");
var _path = _interopRequireDefault(require("path"));
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const AUTH_TYPE_NAMES = exports.AUTH_TYPE_NAMES = {
  BASIC: 'basicauth',
  OIDC: 'oidc',
  JWT: 'jwt',
  SAML: 'saml'
};
class AuthManager {
  constructor({
    kibanaCore,
    sessionStorageFactory,
    pluginDependencies,
    logger,
    searchGuardBackend,
    configService,
    spacesService
  }) {
    /**
     * This needs to be the very first onPreAuth handler that
     * we register for the plugin
     * @param request
     * @param response
     * @param toolkit
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "onPreAuth", async (request, response, toolkit) => {
      if (request.headers.authorization) {
        const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
        const authInstance = await this.getAuthInstanceByCookie({
          request
        });
        if (sessionCookie.credentials && authInstance) {
          // In case we already had a session BEFORE we encountered a request
          // with auth headers, we may need to clear the cookie.
          // Make sure to clear any auth related cookie info if we detect a different header
          await authInstance.clear(request, true);
        }
      }
      return toolkit.next();
    });
    (0, _defineProperty2.default)(this, "checkAuth", async (request, response, toolkit) => {
      const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};

      /**
       * If we have an auth header, just let the request pass through
       */
      if (request.headers.authorization) {
        return toolkit.next();
      }
      try {
        if (request.route.options.authRequired === false) {
          return toolkit.next();
        }
      } catch (error) {
        this.logger.info('Could not read auth options for the path: ' + request.url.pathname);
      }

      /**
       * This block isn't really needed since we started checking
       * the route's auth options, but leaving it in case there
       * are any edge cases
       */
      if (this.routesToIgnore.includes(request.url.pathname) || request.url.pathname.indexOf('/bundle') > -1) {
        return toolkit.next();
      }

      /**
       * This block isn't really needed since we started checking
       * the route's auth options, but leaving it in case there
       * are any edge cases.
       * These routes are used configurable.
       */
      if (this.unauthenticatedRoutes.includes(request.url.pathname)) {
        // If we do this, we don't really assign any relevant headers
        // Until now, we got the kibana server user here, but those credentials were
        // not really used, it seems

        return toolkit.next();
      }
      const authInstanceByRequest = await this.getAuthInstanceByRequest({
        request
      });
      if (authInstanceByRequest) {
        return authInstanceByRequest.checkAuth(request, response, toolkit);
      }

      // @todo This way of handling anonymous auth unfortunately
      // doesn't provide a good way of showing an error message
      // if the SG backend hasn't been configured properly
      if (!sessionCookie.authType && this.configService.get('searchguard.auth.anonymous_auth_enabled')) {
        /*
        return toolkit.authenticated({
        });
        */

        return toolkit.next();
      }

      // Here comes the not authenticated logic...

      // We don't have any cookie, but we may have an optional auth
      try {
        if (request.route.options.authRequired === 'optional') {
          return toolkit.next();
        }
      } catch (error) {
        this.logger.info('Could not read auth options for the path: ' + request.url.pathname);
      }
      const isAjaxRequest = request.headers && (request.headers.accept && request.headers.accept.split(',').indexOf('application/json') > -1 || request.headers['content-type'] && request.headers['content-type'].indexOf('application/json') > -1);
      const nextUrl = this.getNextUrl(request);
      let loginPageURL = this.basePath + '/searchguard/login' + `?nextUrl=${nextUrl}`;
      try {
        const authConfig = await this.searchGuardBackend.getAuthConfig(nextUrl);
        let config;
        if (authConfig && authConfig.auth_methods && authConfig.auth_methods.length == 1 && authConfig.auth_methods[0].sso_location) {
          // If there is only one auth_method with sso_location
          config = authConfig.auth_methods[0];
        } else {
          // If one of the methods has auto_select property enabled
          config = authConfig && authConfig.auth_methods && authConfig.auth_methods.find(({
            auto_select
          }) => auto_select);
        }
        if (config && config.sso_location) {
          loginPageURL = config.sso_location;
          const authInstance = this.authInstances[config.method];
          if (authInstance && authInstance.loginURL) {
            loginPageURL = new URL(this.basePath + authInstance.loginURL, 'http://abc');
            if (config.id) {
              loginPageURL.searchParams.set('authTypeId', config.id);
            }
            if (nextUrl) {
              loginPageURL.searchParams.set('nextUrl', nextUrl);
            }
            loginPageURL = loginPageURL.href.replace(loginPageURL.origin, '');
          }
          if (config.capture_url_fragment && nextUrl && !isAjaxRequest) {
            return response.redirected({
              headers: {
                'location': `${this.basePath}/auth/captureurlfragment?loginHandler=${this.basePath + authInstance.loginURL}&authTypeId=${config.id}&nextUrl=${encodeURIComponent(nextUrl)}`
              }
            });
          }
        }
      } catch (error) {
        console.error("Error while retrieving auth config", error);
      }
      if (isAjaxRequest) {
        // If the session has expired, we may receive ajax requests that can't handle a 302 redirect.
        // In this case, we trigger a 401 and let the interceptor handle the redirect on the client side.
        return response.unauthorized({
          headers: {
            sg_redirectTo: loginPageURL
          },
          body: {
            message: 'Session expired or invalid username and password'
          }
        });
      }
      return response.redirected({
        headers: {
          location: loginPageURL
        }
      });
    });
    /**
     * Handler for validating the auth state in routes with authRequired === optional.
     *
     * Our auth logic runs in the onPreAuth lifecycle step, and
     * we let requests with optional auth pass through.
     *
     * Subsequently, Kibana's auth logic will consider the
     * request authenticated in the default auth logic.
     *
     * If the optional route's handler behaves differently
     * for authenticated request and makes calls to the
     * backend, we may run into unexpected 401s.
     *
     * This handler tries to sync the authentication
     * status with the rest of our auth logic.
     * If there's no authorization header attached to the
     * request, we will consider it unauthenticated.
     *
     * @see https://github.com/elastic/kibana/blob/6c438b331c703c507af524a13aab634cdbfbbb13/dev_docs/tutorials/endpoints.mdx?plain=1#L399
     *
     * @param request
     * @param response
     * @param toolkit
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "handleAuthForOptionalRoutes", async (request, response, toolkit) => {
      try {
        if (request.route.options.authRequired === 'optional' && !request.headers.authorization) {
          const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
          rawRequest.auth.isAuthenticated = false;
        }
      } catch (error) {
        this.logger.info('handleAuthForOptionalRoute could not read auth options for the path: ' + request.url.pathname);
      }
      return toolkit.next();
    });
    /**
     * Handler for validating the auth state in routes with authRequired === optional.
     *
     * Our auth logic runs in the onPreAuth lifecycle step, and
     * we let requests with optional auth pass through.
     *
     * Subsequently, Kibana's auth logic will consider the
     * request authenticated in the default auth logic.
     *
     * If the optional route's handler behaves differently
     * for authenticated request and makes calls to the
     * backend, we may run into unexpected 401s.
     *
     * This handler tries to sync the authentication
     * status with the rest of our auth logic.
     * If there's no authorization header attached to the
     * request, we will consider it unauthenticated.
     *
     * @see https://github.com/elastic/kibana/blob/6c438b331c703c507af524a13aab634cdbfbbb13/dev_docs/tutorials/endpoints.mdx?plain=1#L399
     *
     * @param request
     * @param response
     * @param toolkit
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "handleAuthForOptionalRoutes", async (request, response, toolkit) => {
      try {
        if (request.auth.isAuthenticated === true && request.route.options.authRequired === 'optional' && !request.headers.authorization) {
          // Set isAuthenticated to false
          if (['/login'].indexOf(request.url.pathname) === -1) {
            const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
            rawRequest.auth.isAuthenticated = false;
          }
        }
      } catch (error) {
        this.logger.info('handleAuthForOptionalRoute could not read auth options for the path: ' + request.url.pathname);
      }
      return toolkit.next();
    });
    // @todo Not needed for 7.10?
    (0, _defineProperty2.default)(this, "onPostAuth", async (request, response, toolkit) => {
      if (request.route.path === '/api/core/capabilities') {
        if (this.configService.get('searchguard.auth.anonymous_auth_enabled')) return toolkit.next();
        const authHeaders = await this.getAllAuthHeaders(request);
        if (authHeaders === false && !request.headers.authorization) {
          /*
          We need this redirect because Kibana calls the capabilities on our login page. The Kibana checks if there is the default space in the Kibana index.
          The problem is that the Kibana call is scoped to the current request. And the current request doesn't contain any credentials in the headers because the user hasn't been authenticated yet.
          As a result, the call fails with 401, and the user sees the Kibana error page instead of our login page.
          We flank this issue by redirecting the Kibana call to our route /api/v1/searchguard/kibana_capabilities where we serve some
          minimum amount of capabilities. We expect that Kibana fetches the capabilities again once the user logged in.
          */
          // The payload is passed together with the redirect despite of the undefined here
          return new _coreHttpServer.KibanaResponse(307, undefined, {
            headers: {
              location: this.basePath + '/api/v1/searchguard/kibana_capabilities'
            }
          });
        }
      }
      return toolkit.next();
    });
    this.kibanaCore = kibanaCore;
    this.sessionStorageFactory = sessionStorageFactory;
    this.searchGuardBackend = searchGuardBackend;
    this.logger = logger;
    this.pluginDependencies = pluginDependencies;
    this.configService = configService;
    this.spacesService = spacesService;
    this.authInstances = {};
    this.unauthenticatedRoutes = this.configService.get('searchguard.auth.unauthenticated_routes');

    /**
     * We used to have a list of paths that auth should ignore.
     * Now, we check the route options instead - if no auth
     * is required, we just skip the path.
     *
     * Keeping this in case there are exceptions to this rule.
     */
    this.routesToIgnore = [];
    this.basePath = kibanaCore.http.basePath.get();
  }
  registerAuthInstances() {
    [require('./types/openid/OpenId'), require('./types/basicauth/BasicAuth'), require('./types/jwt/Jwt'), require('./types/saml/Saml')].forEach(AuthClass => {
      // @todo This needs to respect the order as given by the backend
      const authInstance = new AuthClass({
        kibanaCore: this.kibanaCore,
        sessionStorageFactory: this.sessionStorageFactory,
        pluginDependencies: this.pluginDependencies,
        logger: this.logger,
        searchGuardBackend: this.searchGuardBackend,
        config: this.configService,
        authManager: this,
        // @todo Is the authManager used?
        spacesService: this.spacesService
      });
      authInstance.init();
      this.authInstances[authInstance.type] = authInstance;
    });
  }
  registerAuthInstance(authTypeName, authInstance) {
    this.authInstances[authTypeName] = authInstance;
  }
  getAuthInstanceByName(authTypeName) {
    if (this.authInstances[authTypeName]) {
      return this.authInstances[authTypeName];
    }
    return null;
  }
  async getAuthInstanceByRequest({
    request
  }) {
    const matchedAuthInstance = await this.getAuthInstanceByAuthTypes({
      request
    });
    // matchedAuthInstance will be null if we didn't get a match
    if (matchedAuthInstance) {
      return matchedAuthInstance;
    }
    const authInstanceByCookie = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstanceByCookie) {
      return authInstanceByCookie;
    }
    return null;
  }
  async getAuthInstanceByAuthTypes({
    request
  }) {
    for (const authType in this.authInstances) {
      const authInstance = this.getAuthInstanceByName(authType);
      const authInstanceResult = await authInstance.detectCredentialsByRequest({
        request
      });
      if (authInstanceResult !== null) {
        return authInstance;
      }
    }
    return null;
  }
  async getAuthInstanceByCookie({
    request
  }) {
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    if (sessionCookie.authType && this.authInstances[sessionCookie.authType]) {
      return this.getAuthInstanceByName(sessionCookie.authType);
    }
    return null;
  }
  getNextUrl(request) {
    let nextUrl = _path.default.posix.join(this.basePath, request.url.pathname);
    if (request.url.search) nextUrl += request.url.search;
    return nextUrl;
  }

  /**
   * Get credentials from an existing cookie only
   * @param request
   * @returns {Promise<*|boolean|boolean>}
   */
  async getAuthHeader(request) {
    const authInstance = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstance) {
      // @todo A bit weird that we have different method signatures here
      const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      return authInstance.getAuthHeader(sessionCookie);
    }
    return false;
  }
  async getAllAuthHeaders(request) {
    if (request.headers.authorization) {
      return false;
    }
    const authInstance = await this.getAuthInstanceByRequest({
      request
    });
    if (authInstance) {
      return authInstance.getAllAuthHeaders(request);
    }
    return false;
  }
  async logout({
    context,
    request,
    response
  }) {
    const authInstance = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstance) {
      return await authInstance.logout({
        context,
        request,
        response
      });
    }
    return response.ok();
  }
}
exports.AuthManager = AuthManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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